import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/legacy.dart';
import 'package:shared_preferences/shared_preferences.dart';

enum FontSizeOption {
  small,
  medium,
  large,
}

class FontSizeBloc extends ChangeNotifier {
  final String key = "font_size";
  SharedPreferences? _pref;
  FontSizeOption _fontSize = FontSizeOption.medium;

  FontSizeOption get fontSize => _fontSize;

  // Get text scale factor based on font size option
  double get textScaleFactor {
    switch (_fontSize) {
      case FontSizeOption.small:
        return 0.85;
      case FontSizeOption.medium:
        return 1.0;
      case FontSizeOption.large:
        return 1.15;
    }
  }

  FontSizeBloc() {
    _loadFromPrefs();
  }

  void setFontSize(FontSizeOption size) {
    _fontSize = size;
    _saveToPrefs();
    notifyListeners();
  }

  Future<void> _initPrefs() async {
    _pref ??= await SharedPreferences.getInstance();
  }

  Future<void> _loadFromPrefs() async {
    await _initPrefs();
    final savedSize = _pref!.getString(key);
    if (savedSize != null) {
      _fontSize = FontSizeOption.values.firstWhere(
        (e) => e.toString() == savedSize,
        orElse: () => FontSizeOption.medium,
      );
    }
    notifyListeners();
  }

  Future<void> _saveToPrefs() async {
    await _initPrefs();
    _pref!.setString(key, _fontSize.toString());
  }
}

final fontSizeBlocProvider = ChangeNotifierProvider<FontSizeBloc>((ref) {
  return FontSizeBloc();
});

